/*********************************************************************************************************//**
 * @file    BMK54T004_HT32/src/BMK54T004.c
 * @version V1.0.1
 * @date    2024-08-06
 * @brief   The function of BMK54T004 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMK54T004.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMK54T004_WIRE = BMK54T004_WIRE;
/* Private variables ---------------------------------------------------------------------------------------*/
/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief module wire number select.
 * @param wire_number: select wire number.
 * @retval BMK54T004_SUCCESS/BMK54T004_FAILURE
 ************************************************************************************************************/
BMK54T004_selStatus BMK54T004_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BMK54T004_FAILURE;
  }

  gBMK54T004_WIRE = wire_number;
  return BMK54T004_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BMK54T004_Init(void)
{
	I2CMaster_Init(gBMK54T004_WIRE, BMK54T004_MASTERADDR, BMK54T004_CLKFREQ);
	
	//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BMK54T004_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BMK54T004_INTPIN_GPIO_ID, BMK54T004_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BMK54T004_INTPIN_GPIO_PORT, BMK54T004_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BMK54T004_INTPIN_GPIO_PORT, BMK54T004_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BMK54T004_INTPIN_GPIO_PORT, BMK54T004_INTPIN_GPIO_PIN, ENABLE);
}

/*********************************************************************************************************//**
 * @brief  get Key Status
 * @param  void
 * @retval Returns the INT state  
             @arg 0:INT output low level  press
             @arg 1:INT output high level   unpress 
 ************************************************************************************************************/
uint8_t BMK54T004_getINT(void)
{
	return (GPIO_ReadInBit(BMK54T004_INTPIN_GPIO_PORT, BMK54T004_INTPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  read Key Value
 * @param  void
 * @retval key value:Variables for storing value Key
                 kvalue=0:No key is pressed
                  @arg bit0=1 : key1 is pressed
                  @arg bit1=1 : key2 is pressed
                  @arg bit2=1 : key3 is pressed
                  @arg bit3=1 : key4 is pressed 
 ************************************************************************************************************/
uint16_t BMK54T004_readKeyValue(void)
{
	  uint8_t BMK54T004_keyValue[1];
    uint8_t BMK54T004_KeyCMD[1]={0X0E};
    BMK54T004_writeBytes(BMK54T004_KeyCMD,1); 
    _BMK54T004_delay(5);
    BMK54T004_readBytes(BMK54T004_keyValue,1);
    _BMK54T004_delay(10);
    return BMK54T004_keyValue[0];
}

/*********************************************************************************************************//**
 * @brief  read Slide Value
 * @param  void
 * @retval slide value:Variables for storing value Slide
                   @arg 0:The slider is untouched
                   @arg n:the  Slide n is touched(1~8)
 ************************************************************************************************************/
uint16_t BMK54T004_readSlideValue(void)
{
	  uint8_t BMK54T004_keyValue[1];
    uint8_t BMK54T004_KeyCMD[1]={0X0D};
    BMK54T004_writeBytes(BMK54T004_KeyCMD,1); 
    _BMK54T004_delay(5);
    BMK54T004_readBytes(BMK54T004_keyValue,1);
    _BMK54T004_delay(10);
    return BMK54T004_keyValue[0];
}

/*********************************************************************************************************//**
 * @brief  get Threshold
 * @param  buff :Store acquired 12 touch threshold 
               buff[0]~buff[7]:Store the threshold of the slider at positions 1 to 8
               buff[8]~buff[11]:Store the threshold of key1 to key4  
 * @retval void
 ************************************************************************************************************/
void BMK54T004_getThreshold(uint8_t buff[])
{
	  uint8_t BMK54T004_TLCMD[1] = {0xD8};//threshold CMD
    BMK54T004_writeBytes(BMK54T004_TLCMD,1); 
    _BMK54T004_delay(5); 
    BMK54T004_readBytes(buff,12); 
    _BMK54T004_delay(10);
}

/*********************************************************************************************************//**
 * @brief  set Threshold
 * @param  buff :Set acquired 12 touch threshold 
               buff[0]~buff[7]:Set the threshold of the slider at positions 1 to 8,range: 10-64
               buff[8]~buff[11]:Set the threshold of key1 to key4,range: 10-64 
 * @retval Implementation status:
              @arg 0:BMK54T004_R_SUCCESS 
              @arg 1:BMK54T004_R_FAIL
 ************************************************************************************************************/
int BMK54T004_setThreshold(uint8_t buff[])
{
    uint16_t BMK54T004_total = 0;
    uint8_t BMK54T004_TXbuff[14] = {0};
		uint8_t i=0;
    for(i=0;i<12;i++)                    //sum
    {
        BMK54T004_total = BMK54T004_total + buff[i];
        BMK54T004_TXbuff[i+1] = buff[i];
    }
    if(160 <= BMK54T004_total && BMK54T004_total<= 1024)//10~64
    {
        BMK54T004_TXbuff[0] = 0XD8;
        BMK54T004_TXbuff[13] = (uint8_t)BMK54T004_total;
        BMK54T004_writeBytes(BMK54T004_TXbuff,14);
        _BMK54T004_delay(10);
        return BMK54T004_R_SUCCESS;
    }
    else
    {
      return BMK54T004_R_FAIL;
    }
}

/*********************************************************************************************************//**
 * @brief  set all Threshold Level
 * @param  level :Storing ThresholdLevel Data,range: 0-6  
 * @retval Implementation status:
              @arg 0:BMK54T004_R_SUCCESS 
              @arg 1:BMK54T004_R_FAIL 
 ************************************************************************************************************/
int BMK54T004_setAllThresholdLevel(uint8_t level)
{
		uint8_t BMK54T004_thr = 0;
		uint8_t BMK54T004_sum =  0;
	  uint8_t BMK54T004_STLCMD[20] = {0};
	  uint8_t i = 0;
    if(level>6)
    {
        return BMK54T004_R_FAIL;
    }
    BMK54T004_thr = 10+level*9;
    BMK54T004_sum = 12*BMK54T004_thr;
		BMK54T004_STLCMD[0] = 0xD8;
		for(i = 1;i<13;i++)
		{
			BMK54T004_STLCMD[i] = BMK54T004_thr;
		}
		BMK54T004_STLCMD[13] = BMK54T004_sum;
    BMK54T004_writeBytes(BMK54T004_STLCMD,14);
    _BMK54T004_delay(10);
    return BMK54T004_R_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  writeBytes
 * @param  wbuf :the bytes sent
             wlen :the length of the data sent  
 * @retval void
 ************************************************************************************************************/
void BMK54T004_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	  uint8_t sendbuf[255] = {0};
		uint8_t i = 0;
		for(i=0;i<wlen;i++)
		{
			sendbuf[i] = wbuf[i];
		}
		
	{
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = BMK54T004_MASTERADDR;
    gI2CMaster.Tx.puBuffer = (uint8_t *)&sendbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;

    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBMK54T004_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBMK54T004_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
	}
}

/*********************************************************************************************************//**
 * @brief  read Bytes
 * @param  rbuf :the bytes receive
             rlen :the length of the data receive
 * @retval void
 ************************************************************************************************************/
void BMK54T004_readBytes(uint8_t rbuf[], uint8_t rlen)
{
     I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
     I2CMaster_Typedef I2CMaster = { 0 };

     I2CMaster.uSlaveAddress = BMK54T004_MASTERADDR;
     I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
     I2CMaster.Rx.uLength = rlen;
     I2CMaster.uTimeout_us = 30000;
     I2CMaster.isDynamicOnOff_I2C = FALSE;

     I2CMaster_Read(gBMK54T004_WIRE, &I2CMaster);
	
     do {
       I2CResult = I2CMaster_GetStatus(gBMK54T004_WIRE);
     } while (I2CResult == I2C_MASTER_GOING);
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BMK54T004_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}


/* Private functions ---------------------------------------------------------------------------------------*/

